#include "imgui.h"
#include <ctime>
#include <string>
#include <cmath>
#include <cstdio>

// 🔹 Expiry datetime (format: YYYY-MM-DD HH:MM:SS)
static const char* EXP_DATE = "2027-12-09 12:00:00"; 

// 🔹 Reliable manual parsing
time_t StringToTime(const char* datetime)
{
    int year, month, day, hour, min, sec;
    if (sscanf(datetime, "%d-%d-%d %d:%d:%d", &year, &month, &day, &hour, &min, &sec) != 6) {
        return 0; // fail safe
    }

    std::tm tm = {};
    tm.tm_year = year - 1900;
    tm.tm_mon  = month - 1; // 0 = Jan
    tm.tm_mday = day;
    tm.tm_hour = hour;
    tm.tm_min  = min;
    tm.tm_sec  = sec;
    tm.tm_isdst = -1;

    return mktime(&tm);
}

// 🔹 Remaining time calculation
std::string GetRemainingTime(time_t expiry)
{
    time_t now = time(nullptr);
    double diff = difftime(expiry, now);

    if (diff <= 0) return "Expired";

    int days  = (int)(diff / (60 * 60 * 24));
    int hours = ((int)diff % (60 * 60 * 24)) / 3600;
    int mins  = ((int)diff % 3600) / 60;
    int secs  = (int)diff % 60;

    char buffer[128];
    if (days > 0)
        snprintf(buffer, sizeof(buffer), "%dD %02dH %02dM %02dS", days, hours, mins, secs);
    else
        snprintf(buffer, sizeof(buffer), "%02dH %02dM %02dS", hours, mins, secs);

    return std::string(buffer);
}

// 🔹 Render top-left license/expiry box
void RenderExpiredBoxTopLeft(const char* title)
{
    ImGuiIO& io = ImGui::GetIO();
    ImDrawList* draw = ImGui::GetBackgroundDrawList();

    int screenWidth  = io.DisplaySize.x;
    int screenHeight = io.DisplaySize.y;

    float fontSize  = 22.0f;
    float paddingX  = 20.0f;
    float paddingY  = 8.0f;

    ImFont* font = io.Fonts->Fonts[0];

    // 🔹 Check expire
    time_t expiryTime = StringToTime(EXP_DATE);
    std::string status = GetRemainingTime(expiryTime);
    bool expired = (status == "Expired");

    std::string fullText = std::string(title) + " | " + status;

    ImVec2 textSize = font->CalcTextSizeA(fontSize, FLT_MAX, -1.0f, fullText.c_str());
    ImVec2 boxSize  = ImVec2(textSize.x + paddingX * 2, textSize.y + paddingY * 2);

    // ✅ Position: top-left (20px margin)
    ImVec2 boxPos = ImVec2(20.0f, 20.0f);
    ImVec2 boxEnd = ImVec2(boxPos.x + boxSize.x, boxPos.y + boxSize.y);

    // 🔹 Colors
    ImU32 boxColor    = expired ? ImColor(50, 0, 0, 220) : ImColor(0, 40, 30, 200);
    ImU32 borderColor = expired ? ImColor(255, 0, 0, 255) : ImColor(0, 255, 120, 255);
    ImU32 textColor   = expired ? ImColor(255, 100, 100, 255) : ImColor(200, 255, 200, 255);

    // Box
    draw->AddRectFilled(boxPos, boxEnd, boxColor, 10.0f);
    draw->AddRect(boxPos, boxEnd, borderColor, 10.0f, 0, 3.0f);

    // Center text
    ImVec2 startPos = ImVec2(
        boxPos.x + (boxSize.x - textSize.x) * 0.5f,
        boxPos.y + (boxSize.y - textSize.y) * 0.5f
    );

    // Glow effect if not expired
    if (!expired)
    {
        float t = ImGui::GetTime() * 2.0f;
        int r = (int)((sin(t) * 0.5f + 0.5f) * 255);
        int g = (int)((sin(t + 2.0f) * 0.5f + 0.5f) * 255);
        int b = (int)((sin(t + 4.0f) * 0.5f + 0.5f) * 255);
        ImColor glowColor = ImColor(r, g, b, 120);

        for (int i = 4; i >= 1; i--)
        {
            draw->AddText(font, fontSize,
                          ImVec2(startPos.x + i * 0.6f, startPos.y + i * 0.6f),
                          glowColor,
                          fullText.c_str());
        }
    }

    // Main text
    draw->AddText(font, fontSize, startPos, textColor, fullText.c_str());
}
